/**
  ******************************************************************************
  * @file    flashloader.c
  * @author  Milandr Application Team
  * @version V1.2.0
  * @date    09/12/2024
  * @brief   Flashloader for 1986VE9x, 1986VE1, 1986VE3, 1901VC1 microcontrollers.
  ******************************************************************************
  * <br><br>
  * THE PRESENT FIRMWARE IS FOR GUIDANCE ONLY. IT AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING MILANDR'S PRODUCTS IN ORDER TO FACILITATE
  * THE USE AND SAVE TIME. MILANDR SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES RESULTING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR A USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN THEIR PRODUCTS.
  *
  * <h2><center>&copy; COPYRIGHT 2024 Milandr</center></h2>
  */

/* Includes ------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "flash_loader.h"
#include "flash_loader_extra.h"

#include "MDR32F9Qx_config.h"
#include "MDR32F9Qx_eeprom.h"
#include "MDR32F9Qx_rst_clk.h"
#include "MDR32F9Qx_utils.h"
#include "MDR32F9Qx_bkp.h"

/* Private function prototypes -----------------------------------------------*/
void BKP_SetDefaultTrimmings(void);

/* Private variables ---------------------------------------------------------*/
#if defined (_USE_DEBUG_UART_) /* Must be declared in MDR32F9Qx_config.h */
static const char debug_uart_err[] = "Debug UART initialization failed!";
#endif
static __no_init uint32_t mass_erase_done;

/* Private functions ---------------------------------------------------------*/
/**
  * @brief  Initializes the flash driver.
  * @param  base_of_flash: Points to the first byte of the whole flash memory range.
  * @param  image_size: Specifies the size of the whole image that is to be written to flash memory, in bytes.
  * @param  link_address: Specifies the original link address of the first byte of the image, before any offsets 
  * @param  flags: Specifies optional flags.
  * @param  argc: Number of arguments. Only used if USE_ARGC_ARGV is set.
  * @param  argv: An array of argument string pointers. Only used if USE_ARGC_ARGV is set.
  * @retval RESULT_OK - init successful; otherwise returns an error code.
  */
#if USE_ARGC_ARGV
uint32_t FlashInit(void *base_of_flash, uint32_t image_size, uint32_t link_address, uint32_t flags, int argc, char const *argv[])
#else
uint32_t FlashInit(void *base_of_flash, uint32_t image_size, uint32_t link_address, uint32_t flags)
#endif
{
    uint32_t result = RESULT_OK;
    mass_erase_done = 0;

    __disable_irq();

    RST_CLK_PCLKcmd(RST_CLK_PCLK_BKP, ENABLE);
    RST_CLK_LSIcmd(DISABLE); /* To stop IWDG */
    BKP_SetDefaultTrimmings();

    RST_CLK_DeInit();
    RST_CLK_PCLKcmd(RST_CLK_PCLK_EEPROM, ENABLE);

#if defined (_USE_DEBUG_UART_)
    if(STDIO_Init() == ERROR)
    {
        strcpy(ERROR_MESSAGE_BUFFER, debug_uart_err);
        return RESULT_ERROR_WITH_MSG;
    }
    printf("FlashInit: base_of_flash = 0x%X, image_size = 0x%X, link_address = 0x%X, flags = 0x%X\n\r", (uint32_t)base_of_flash, image_size, link_address, flags);
#endif /* _USE_DEBUG_UART_ */

    if(flags & (FLAG_ERASE_ONLY | FLAG_MASS_ERASE))
    {
        EEPROM_EraseAllPages(EEPROM_Main_Bank_Select);
        if(flags & FLAG_ERASE_ONLY)
        {
            result = RESULT_ERASE_DONE;
        }
        else
        {
            mass_erase_done = 1;
        }
    }

#if defined (_USE_DEBUG_UART_)
    printf("FlashInit done! Result = 0x%X\n\r", result);
#endif /* _USE_DEBUG_UART_ */

    return result;
}

/**
  * @brief  Writes a data buffer in the internal flash.
  * @param  block_start: Points to the first byte of the block into which this write operation writes.
  * @param  offset_into_block: Specifies how far into the current block that this write operation shall start.
  * @param  count: Specifies the number of bytes to write.
  * @param  buffer: A pointer to the buffer that contains the bytes to write.
  * @retval RESULT_OK - write successful; RESULT_ERROR - write fail.
  */
uint32_t FlashWrite(void *block_start, uint32_t offset_into_block, uint32_t count, char const *buffer)
{
    uint32_t size = 0;

#if defined (_USE_DEBUG_UART_)
    printf("FlashWrite: block_start = 0x%X, offset_into_block = 0x%X, count = 0x%X, buffer = 0x%X\n\r", (uint32_t)block_start, offset_into_block, count, (uint32_t)buffer);
#endif /* _USE_DEBUG_UART_ */

    while(size < count)
    {
        EEPROM_ProgramWord((uint32_t)block_start + offset_into_block + size, EEPROM_Main_Bank_Select, *((uint32_t *)buffer));
        buffer += 4;
        size += 4;
    }

#if defined (_USE_DEBUG_UART_)
    printf("FlashWrite done! Result = 0x%X\n\r", RESULT_OK);
#endif /* _USE_DEBUG_UART_ */

    return RESULT_OK;
}

/**
  * @brief  Erase block in the internal flash.
  * @param  block_start: Points to the first byte of the block to erase.
  * @param  block_size: Specifies the size of the block, in bytes.
  * @retval RESULT_OK - erase successful; RESULT_ERROR - erase fail.
  */
uint32_t FlashErase(void *block_start, uint32_t block_size)
{
#if defined (_USE_DEBUG_UART_)
    printf("FlashErase: block_start = 0x%X, block_size = 0x%X, mass_erase_done = 0x%X\n\r", (uint32_t)block_start, block_size, mass_erase_done);
#endif /* _USE_DEBUG_UART_ */

    if(mass_erase_done == 0)
    {
        EEPROM_ErasePage((uint32_t)block_start, EEPROM_Main_Bank_Select);
    }

#if defined (_USE_DEBUG_UART_)
    printf("FlashErase done! Result = 0x%X\n\r", RESULT_OK);
#endif /* _USE_DEBUG_UART_ */

    return RESULT_OK;
}

/**
  * @brief  Calculates the checksum of the downloaded contents of flash memory.
  * @param  begin: Points to the first byte of the block to calculate the checksum.
  * @param  count: Specifies the size of the block, in bytes.
  * @retval calculated checksum value.
  */
OPTIONAL_CHECKSUM
uint32_t FlashChecksum(void const *begin, uint32_t count)
{
    uint16_t crc16 = 0;

#if defined (_USE_DEBUG_UART_)
    printf("FlashChecksum: begin = 0x%X, count = 0x%X\n\r", (uint32_t)begin, count);
#endif /* _USE_DEBUG_UART_ */

    EEPROM_UpdateDCache();
    crc16 = Crc16((uint8_t const *)begin, count);

#if defined (_USE_DEBUG_UART_)
    printf("FlashChecksum done! CRC16 = 0x%X\n\r", crc16);
#endif /* _USE_DEBUG_UART_ */

    return crc16;
}

/**
  * @brief  Clearing after the flash is loaded. The function is called after the last call to FlashWrite 
  *         or after FlashChecksum if it exists.
  * @param  None
  * @retval RESULT_OK
  */
OPTIONAL_SIGNOFF
uint32_t FlashSignoff()
{
#if defined (_USE_DEBUG_UART_)
    printf("FlashSignoff\n\r");
#endif /* _USE_DEBUG_UART_ */

    EEPROM_UpdateDCache();
    RST_CLK_PCLKcmd(RST_CLK_PCLK_EEPROM, DISABLE);

    RST_CLK_PCLKcmd(RST_CLK_PCLK_BKP, ENABLE);
    RST_CLK_LSIcmd(ENABLE);
    RST_CLK_PCLKcmd(RST_CLK_PCLK_BKP, DISABLE);

    __enable_irq();

#if defined (_USE_DEBUG_UART_)
    printf("FlashSignoff done! Result = 0x%X\n\r", RESULT_OK);
#endif /* _USE_DEBUG_UART_ */

    return RESULT_OK;
}

/**
  * @brief  Set LDO, HSI and LSI trimmings to their default reset values.
  * @param  None
  * @retval None
  */
void BKP_SetDefaultTrimmings(void)
{
#if defined (USE_MDR1986VE1T) || defined (USE_MDR1986VE3)
    BKP_SetTrim(BKP_TRIM_1_8_V);
#endif
    BKP_DUccTrim(BKP_DUcc_plus_100mV);
    RST_CLK_HSIadjust(0x20);
    RST_CLK_LSIadjust(0x10);
}

